<?php  if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage database
* @filesource
*/

require_once APPPATH.'libraries/Validator.php';

/**
* Patches and extensions to the CI SQL Server database driver go here.
*
* The ability to extend database drivers is not native to Codeigniter; this extension relies on the Loader class being extended to look for database drivers.
* For more information, see {@link https://github.com/EllisLab/CodeIgniter/wiki/Extending-Database-Drivers Extending Database Drivers}.
*
* @author M. Gibbs <gibbs_margaret@bah.com>
*
* @package direct-project-innovation-initiative
* @subpackage database
*/
class DPII_DB_sqlsrv_driver extends CI_DB_sqlsrv_driver {

	protected $_affected_rows; //the PHP function get affected rows can only be called once, so we'll store the value here so we can access it multiple tiems
	var $_escape_char = '[';
	var $_escape_char_close = ']';
	
	function __construct($params){
		parent::__construct($params);
		log_message('debug', 'Extended DB driver class instantiated.');
	}	
	
	/**
	* Overrides parent to accomodate changes to {@link affected_rows()}.
	* The parent method makes all queries scrollable, but we can't get the {@link affected_rows()} value for scrollable queries.
	* Adjusted so that UPDATE, INSERT, and DELETE queries are not scrollable. 
	*/
	function _execute($sql){
		$this->_affected_rows = null; //reset the _affected_rows count before running the new query
		
		$sql = $this->_prep_query($sql);

		//don't use scrollable param for anything that we might call affected_rows() for
		if(preg_match('/\b(INSERT|UPDATE|DELETE)\b/i', $sql)){
			return sqlsrv_query($this->conn_id, $sql);
		}

		return sqlsrv_query($this->conn_id, $sql, null, array(
			'Scrollable'				=> SQLSRV_CURSOR_STATIC,
			'SendStreamParamsAtExec'	=> true
		));
	}
	
	/** 
	* Overrides parent to correct fatal error.
	* CI mispelled the name of the sqlsrv function in their version of this method, causing a fatal error.  Good job, CI.  In addition to correcting the fatal error,
	* this method will trigger warnings if anything happens that prevents it from calculating the affected rows, instead of failing silently. Also makes use of a class
	* var to store the number of affected rows so that we can safely call this method more than once per query.
	*/
	function affected_rows(){
		//affected rows is only available once, so save it to a protected var so that we can access it multiple times
		if(!isset($this->_affected_rows) || is_null($this->_affected_rows)){
			$this->_affected_rows = sqlsrv_rows_affected($this->result_id);
			if($this->_affected_rows === FALSE){
				foreach(sqlsrv_errors() as $error){
					trigger_error('Error encountered while calling sqlsrv_rows_affected(): '.$error['message'], E_USER_WARNING);
				}
			}
		}
			
		return $this->_affected_rows;
	}	
	
	/** 
	* Overrides parent to escape all strings with N'
	* Our default character set is UTF-8, but the database will only store UTF-8 strings correctly if we prefix string values
	* with N' and make sure that we use nvarchar/nchar/ntext fields instead varchar/char/test fields.  We are now assuming
	* that *all* of our text fields in the database are going to be nvarchar/nchar/ntext fields and automatically prefixing
	* all escaped strings accordingly; it's up to the developers to continue implementing the correct database field types.
	* @param string
	* @return string
	*/
	/*function escape($value){
		$value = parent::escape($value);
		if(is_string($value) && string_begins_with("'", $value) && !is_numeric($value))
			$value = 'N'.$value;
		return $value;
	} */ //written for api - need to double-check that this won't be a problem for non-nvarchar values before enabling this for webmail -- MG 2014-08-22

	 /**
     * An attempt to make _limit() work the same way for SQL Server that it would for more enlightened databases that allow offsets.
	 *
	 * Note that using this method with a non-empty offset will add an additional column called __daas_row_number to your database results.  
	 *
	 * Also
	 *
	 * Note that this method should be revised should we ever upgrade to SQL Server 2012, which has a better way of doing this which wo
     */
    function _limit($sql, $limit, $offset){
		
		if(empty($offset) || !is_numeric($offset) || $offset <= 0){
			return preg_replace('/(^\SELECT (DISTINCT)?)/i','\\1 TOP '.$limit.' ', $sql);     
		}	
		
		//HACKING A WAY TO DO OFFSETS IN SQL SERVER
		//Since SQL server < 2012 doesn't have a good way to do mysql offset, we're going to use the ROW_NUMBER OVER syntax to create row numbers in a derived table, 
		// and select the row numbers we need.  This may go poorly.  
		//Questions?  Ask Margaret.  She will probably deny all knowledge of ever having tried to cobble together a solution for this problem, but you never know.
			
		//hokay.  we need to figure out how this select is ordered, and remove the existing order by if it's there.  conveniently, order by is always the last part of the query if it exists.
		$order_by_position = mb_strpos($sql, 'ORDER BY ');
		if(empty($order_by_position)){
			//if they haven't a supplied an order by, we just can't apply the offset.  Let the developer know that they made a mistake.
			return $this->display_error( array('An ORDER BY value must be applied before a LIMIT offset can be applied to this query:', $sql), '', TRUE );
		}else{
			$order_by = trim(mb_substr($sql, $order_by_position - 1)); //grab the $order_by for our use
			$sql = mb_substr($sql, 0, $order_by_position); //remove the order by clause so that we can use it in ROW_NUMBER() instead
		}

		//now we need to add an additional column to the select query.  
		//the keyword after the columns for select is either INTO or FROM, so figure out where the select ends
		$end_of_select = mb_strpos($sql, 'FROM ') - 1;
		$into_position = mb_strpos($sql, 'INTO ');
		if(!empty($into_position) && $into_position < $end_of_select)
			$end_of_select = $into_position - 1;

		//note that because of the difference between our local SQL version (2008 R2) and the test/prod version (SQL Server 2008 SP3), we need to make sure we explicitly order by row number
		//see: http://dba.stackexchange.com/questions/27467/is-select-row-number-guaranteed-to-return-results-sorted-by-the-generated-row and ticket VAD-732
		$row_number_column = '__daas_row_number'; //ridiculous name, but descriptive & no one else is likely to need to use it		
		$sql = mb_substr($sql, 0, $end_of_select).', ROW_NUMBER() OVER ('.$order_by.') AS '.$row_number_column.mb_substr($sql, $end_of_select);
		$sql = 'SELECT * FROM ('."\n".trim($sql)."\n".') AS __daas_derived_table '."\n".'WHERE '.$row_number_column.' BETWEEN '.($offset+1).' AND '.($offset+$limit).' ORDER BY __daas_row_number';
		return $sql; //VICTORY.
    }

	/**
	 * Escape the SQL Identifiers
	 *
	 * This function escapes column and table names.  Extended because CI assumes that there is a single escape char (like ` in mysql), but mssql uses [].
	 *
	 * @param	string
	 * @return	string
	 */
	function _escape_identifiers($item)
	{		
		if(empty($this->_escape_char))	return $item;
		
		//handle expressions where there's no space between column name and equals sign
		if(string_contains('=', $item)){
			$position_of_equals_sign = mb_strpos($item, '=');
			return $this->_escape_identifiers( mb_substr($item, 0, $position_of_equals_sign) ). mb_substr($item, $position_of_equals_sign) ;			
		}

		
		foreach ($this->_reserved_identifiers as $id)
		{
			
			if(string_contains('.'.$id, $item)){
				$str = $this->_escape_char. str_replace('.', $this->_escape_char_close.'.', $item);
				
				// remove duplicates if the user already included the escape		
				return $this->_remove_duplicate_escape_chars($str);
			}
		}

		if(string_contains('.', $item))
			$str = $this->_escape_char.str_replace('.', $this->_escape_char_close.'.'.$this->_escape_char, $item).$this->_escape_char_close;
		else
			$str = $this->_escape_char.$item.$this->_escape_char_close;
		

		// remove duplicates if the user already included the escape
		return $this->_remove_duplicate_escape_chars($str);
	}


	function _remove_duplicate_escape_chars($string){
		$escape_chars = array($this->_escape_char, $this->_escape_char_close);
		foreach($escape_chars as $escape_char){
			while(string_contains($escape_char.$escape_char, $string))
				$string = str_replace($escape_char.$escape_char, $escape_char, $string);
		}
		return $string;
	}
}
/* End of file mssql_driver.php */
/* Location: ./system/database/drivers/mssql/mssql_driver.php */